#
# ***** BEGIN LICENSE BLOCK *****
# Zimbra Collaboration Suite Server
# Copyright (C) 2010, 2011, 2012, 2013, 2014, 2015, 2016 Synacor, Inc.
#
# This program is free software: you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software Foundation,
# version 2 of the License.
#
# This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
# without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
# See the GNU General Public License for more details.
# You should have received a copy of the GNU General Public License along with this program.
# If not, see <https://www.gnu.org/licenses/>.
# ***** END LICENSE BLOCK *****
#


from base64 import b64decode
import localconfig
import commands
import globalconfig
import miscconfig
import serverconfig
import mtaconfig
from logmsg import *
import config
import shutil
import tempfile
import re
import threading
import time
import traceback
import ldap
import os

MAPPEDFILES = {
	"zimbraSSLDHParam": "conf/dhparam.pem"
}

class State:
	
	lConfig = threading.Lock()
	lAction = threading.Condition()
	mState = None

	startorder = {
		"ldap"      : 0,
		"dnscache"  : 10,
		"logger"    : 20,
		"convertd"  : 30,
		"mailbox"   : 40,
		"mailboxd"  : 50,
		"memcached" : 60,
		"proxy"     : 70,
		"antispam"  : 80,
		"antivirus" : 90,
		"cbpolicyd" : 100,
		"amavis"    : 110,
		"opendkim"  : 120,
		"archiving" : 130,
		"snmp"      : 140,
		"spell"     : 150,
		"mta"       : 160,
		"sasl"      : 170,
		"stats"     : 180,
		}

	def __init__(self):
		self.baseDir          = "/opt/zimbra"
		self.pidFile          = "/opt/zimbra/log/zmconfigd.pid"
		self.hostname         = None
		self.firstRun         = True
		self.forced           = False
		self.localconfig      = localconfig.LocalConfig()
		self.globalconfig     = globalconfig.GlobalConfig()
		self.miscconfig       = miscconfig.MiscConfig()
		self.serverconfig     = serverconfig.ServerConfig()
		self.ldap             = ldap.Ldap()
		self.forcedconfig     = {}
		self.requestedconfig  = {}
		self.fileCache        = {}
		self.watchdogProcess  = {}
		self.changedkeys      = {}
		self.lastVals         = {}
		self.previous         = {
								"rewrites"   : {},
								"config"     : {},
								# "restarts"   : {}, Don't need this, I think
								"postconf"   : {},
								"postconfd"   : {},
								"services"   : {},
								"ldap"       : {},
								"proxygen"   : False # 0|1
							  }
		self.current          = {
								"rewrites"   : {},
								"config"     : {},
								"restarts"   : {},
								"postconf"   : {},
								"postconfd"   : {},
								"services"   : {},
								"ldap"       : {},
								"proxygen"   : False # 0|1
							  }

		self.mtaconfig         = mtaconfig.MtaConfig()
		self.maxFailedRestarts = 3;
		if self.localconfig["zmconfigd_max_failed_restarts"] is not None:
			self.maxFailedRestarts = int(self.localconfig["zmconfigd_max_failed_restarts"])

	def isFalseValue(self,val):
		return (not val or re.match(r"no|false|0+",str(val),re.I))

	def isTrueValue(self,val):
		return not self.isFalseValue(val)

	def delWatchdog(self, process):
		try:
			del self.watchdogProcess[process]
		except:
			return

	def getWatchdog(self, service, state=None):
		if state is not None:
			self.watchdogProcess[service] = state
		try:
			return self.watchdogProcess[service]
		except Exception, e:
			return None

	def proxygen(self, val=None):
		if val is not None:
			self.current["proxygen"] = val
		return self.current["proxygen"]

	def delRestart(self, service):
		try:
			del self.current["restarts"][service]
		except Exception:
			return

	def delLdap(self, service):
		try:
			del self.current["ldap"][service]
		except Exception:
			return

	def clearPostconf(self):
		try:
			self.current["postconf"] = {}
		except Exception:
			return

	def delPostconf(self, service):
		try:
			del self.current["postconf"][service]
		except Exception:
			return

	def clearPostconfd(self):
		try:
			self.current["postconfd"] = {}
		except Exception:
			return

	def delPostconfd(self, service):
		try:
			del self.current["postconfd"][service]
		except Exception:
			return

	def delRewrite(self, service):
		try:
			del self.current["rewrites"][service]
		except Exception:
			return

	def curRewrites(self, service=None, state=None):
		# state should be a tuple (val, mode)
		if service is not None:
			if state is not None:
				Log.logMsg(5, "Adding rewrite %s" % (service,))
				self.current["rewrites"][service] = state
			try:
				return self.current["rewrites"][service]
			except Exception, e:
				return None
		return self.current["rewrites"]

	def curRestarts(self, service=None, state=None):
		if service is not None:
			if state is not None:
				self.current["restarts"][service] = state
			try:
				return self.current["restarts"][service]
			except Exception, e:
				return None
		return self.current["restarts"]

	def curLdap(self, key=None, val=None):
		if key is not None:
			if val is not None:
				Log.logMsg(5, "Adding ldap %s = %s" % (key, val))
				self.current["ldap"][key] = val
			try:
				return self.current["ldap"][key]
			except Exception, e:
				return None
		return self.current["ldap"]

	def curPostconf(self, key=None, val=None):
		if key is not None:
			if val is not None:
				if val == True:
					val = "yes"
				elif val == False:
					val = "no"
				Log.logMsg(5, "Adding postconf %s = %s" % (key, val))
				self.current["postconf"][key] = val.replace('\n', ' ')
			try:
				return self.current["postconf"][key]
			except Exception, e:
				return None
		return self.current["postconf"]

	def curPostconfd(self, key=None, val=None):
		if key is not None:
			if val is not None:
				if val == True:
					val = "yes"
				elif val == False:
					val = "no"
				Log.logMsg(5, "Adding postconfd %s = %s" % (key, val))
				self.current["postconfd"][key] = val.replace('\n', ' ')
			try:
				return self.current["postconfd"][key]
			except Exception, e:
				return None
		return self.current["postconfd"]

	def curServices(self, service=None, state=None):
		if service is not None:
			if state is not None:
				self.current["services"][service] = state
			try:
				return self.current["services"][service]
			except Exception, e:
				return None
		return self.current["services"]

	def prevServices(self, service=None, state=None):
		if service is not None:
			if state is not None:
				self.previous["services"][service] = state
			try:
				return self.previous["services"][service]
			except Exception, e:
				return None
		return self.previous["services"]

	def getAllConfigs(self, cf=None):
		t1 = time.clock()

		Log.logMsg(3, "Fetching All configs")
		# These loading commands really aren't reentrant safe, so we'll need to make sure
		# we only access them from here.  Any rewrite request via interrupt shouldn't require
		# a config reload, anyway.
		Log.logMsg (5, "LOCK myState.lConfig requested")
		State.lConfig.acquire()
		Log.logMsg (5, "LOCK myState.lConfig acquired")
		commands.Command.resetProvisioning("config")
		commands.Command.resetProvisioning("server")
		commands.Command.resetProvisioning("local")
		self.fileCache = {}

		# set thread_wait_time considering ldap_read_timeout
		# in case a connection with ldap is established but the ldap doesn't send any response
		ldap_read_timeout = int(self.localconfig["ldap_read_timeout"])
		thread_wait_time = 60
		if ldap_read_timeout >= 60000:
			thread_wait_time = ldap_read_timeout / 1000

		lc = threading.Thread(target=State.getLocalConfig,args=(self,cf),name="lc")
		gc = threading.Thread(target=State.getGlobalConfig,args=(self,),name="gc")
		mc = threading.Thread(target=State.getMiscConfig,args=(self,),name="mc")
		sc = threading.Thread(target=State.getServerConfig,args=(self,),name="sc")
		lc.setDaemon(True);
		gc.setDaemon(True);
		mc.setDaemon(True);
		sc.setDaemon(True);
		lc.start()
		gc.start()
		mc.start()
		sc.start()
		
		lc.join(thread_wait_time)
		gc.join(thread_wait_time)
		mc.join(thread_wait_time)
		sc.join(thread_wait_time)
		try:
			if (lc.isAlive()):
				Log.logMsg(1, "Thread %s still alive, waiting %d" % (lc.getName(),thread_wait_time))
				lc.join(thread_wait_time)
				if (lc.isAlive()):
					Log.logMsg(1, "Thread %s still alive, aborting" % lc.getName())
					raise Exception, "Thread %s still alive, aborting" % lc.getName()
			if (gc.isAlive()):
				Log.logMsg(1, "Thread %s still alive, waiting %d" % (gc.getName(),thread_wait_time))
				gc.join(thread_wait_time)
				if (gc.isAlive()):
					Log.logMsg(1, "Thread %s still alive, aborting" % gc.getName())
					raise Exception, "Thread %s still alive, aborting" % gc.getName()
			if (mc.isAlive()):
				Log.logMsg(1, "Thread %s still alive, waiting %d" % (mc.getName(),thread_wait_time))
				mc.join(thread_wait_time)
				if (mc.isAlive()):
					Log.logMsg(1, "Thread %s still alive, aborting" % mc.getName())
					raise Exception, "Thread %s still alive, aborting" % mc.getName()
			if (sc.isAlive()):
				Log.logMsg(1, "Thread %s still alive, waiting %d" % (sc.getName(),thread_wait_time))
				sc.join(thread_wait_time)
				if (sc.isAlive()):
					Log.logMsg(1, "Thread %s still alive, aborting" % sc.getName())
					raise Exception, "Thread %s still alive, aborting" % sc.getName()

		except Exception, e:
			State.lConfig.release()
			Log.logMsg (3, "LOCK myState.lConfig released")
			raise e

		State.lConfig.release()
		Log.logMsg (5, "LOCK myState.lConfig released")

		dt = time.clock() - t1
		Log.logMsg(3, "All configs fetched in %.2f seconds" % (dt,))

	def getLocalConfig(self, cf = None):
		self.localconfig.load()
		self.hostname = self.localconfig["zimbra_server_hostname"]
		if cf:
			cf.setVals(self)

	def getGlobalConfig(self):
		self.globalconfig.load()

	def getMiscConfig(self):
		self.miscconfig.load()

	def getServerConfig(self):
		self.serverconfig.load(self.hostname)

	def getMtaConfig(self,cf=None):
		self.mtaconfig.load(cf,self)

	def lookUpConfig(self, type, key):
		if re.match(r"!",key):
			return self.checkConditional(type, key)
		Log.logMsg(5, "Looking up key=%s with type=%s" % (key, type))
		
		value = None

		if type == "VAR":
			if key in self.globalconfig:
				value = self.globalconfig[key]
			if key in self.miscconfig:
				value = self.miscconfig[key]
			if key in self.serverconfig:
				value = self.serverconfig[key]
		elif type == "LOCAL":
			value = self.localconfig[key]
		elif type == "FILE":
			# Do this all in memory
			if not key in self.fileCache:
				tmpfile = os.path.join(self.baseDir,"conf", key)
				lines = [self.transform(l).strip() for l in open(tmpfile,'r').readlines() if self.transform(l).strip()]
				value = ', '.join(lines)
				self.fileCache[key] = value
				Log.logMsg(5, "Loaded %s = %s" % (key, value))
			else:
				value = self.fileCache[key]
				Log.logMsg(5, "Loaded from cache %s = %s" % (key, value))
		elif type == "MAPFILE":
			if key in MAPPEDFILES:
				try:
					val = self.lookUpConfig("VAR", key)
					mapfile = os.path.join(self.baseDir, MAPPEDFILES[key])
					if os.path.isfile(mapfile):
						mapdata = file(mapfile).read()
					else:
						mapdata = None
					if val is not None:
						val = b64decode(val)
						if val != mapdata:
							mf = open(mapfile, "w")
							try:
								mf.write(val)
								mf.flush()
							finally:
								mf.close()
						value = mapfile
					elif mapdata is not None:
						os.remove(mapfile)
						value = None
					Log.logMsg(5, "MAPFILE %s='%s', exists?=%s" % (key, value, os.path.exists(mapfile)))
				except Exception, e:
					Log.logMsg(2, "Error processing (%s, %s) = %s" % (key, value, e))
					value = None
			else:
				Log.logMsg(2, "%s: key '%s' not in MAPPEDFILES" % (type, key))
				value = None
		elif type == "MAPLOCAL":
			if key in MAPPEDFILES:
				mapfile = os.path.join(self.baseDir, MAPPEDFILES[key])
				try:
					if os.path.isfile(mapfile):
						value = mapfile
					else:
						value = ''
				except Exception, e:
					Log.logMsg(2, "Error processing (%s, %s) = %s" % (key, value, e))
					value = None
				Log.logMsg(5, "MAPLOCAL %s='%s', exists?=%s" % (key, value, os.path.exists(mapfile)))
			else:
				Log.logMsg(2, "%s: key '%s' not in MAPPEDFILES" % (type, key))
				value = None
		elif type == "SERVICE":
			value = self.serverconfig.getServices(key)
		else:
			Log.logMsg(2, "Unknown config type %s for key %s" % (type, key))

		Log.logMsg(5, "Looked up key=%s with type=%s value=%s" % (key, type, value))
		return value

	def checkConditional(self, type, key):
		negate = False
		Log.logMsg(5, "Conditional Entry: key=%s type=%s negate=%s" % (key, type, negate))
		if re.match(r'!', key):
			negate = True
		key = re.sub(r'^!','',key)
		Log.logMsg(5, "Conditional After Negate Check: key=%s type=%s negate=%s" % (key, type, negate))
		value = self.lookUpConfig(type, key)
		Log.logMsg(5, "Conditional After lookUpConfig: key=%s val=%s type=%s negate=%s" % (key, value, type, negate))
		Log.logMsg(5, "Checking conditional for negate=%s type=%s %s=%s" % (negate, type, key, value))
		if (self.isFalseValue(value)):
			rvalue = False
		else:
			rvalue = True
		if negate:
			rvalue = not rvalue
		Log.logMsg(5, "Checking conditional for negate=%s type=%s %s=%s return=%s" % (negate, type, key, value, rvalue))
		return rvalue

	def isrunning(self):
		cpid = self.getpid()
		if cpid:
			# python throws if pid does't exist, jython returns -1
			try:
				if (os.kill(cpid,0) != -1):
					Log.logMsg(1, "zmconfigd already running at %d" % (cpid,)) 
					return True
			except:
				return False
		return False

	def getpid(self):
		pf = self.pidFile
		try:
			cpid = open(pf).readline().strip()
		except Exception, e:
			return 0
		return int(cpid)

	def writepid(self):
		try:
			pid = str(os.getpid())
			Log.logMsg(4, "Writing %s to %s" % (pid, self.pidFile))
			f = open(self.pidFile, 'w+')
			f.write("%s\n" % (pid,))
		except Exception, e:
			[Log.logMsg(1,t) for t in traceback.format_tb(sys.exc_info()[2])]
			Log.logMsg (0, "writepid() failed: %s" % (e,))
		finally:
			f.close()

	def resetChangedKeys(self, section):
		self.changedkeys[section] = []

	def changedKeys(self, section, key=None):
		if not section in self.changedkeys:
			self.resetChangedKeys(section)
		if key is not None:
			self.changedkeys[section].append(key)
		return self.changedkeys[section]

	def delVal(self, section, type, key):
		try:
			del self.lastVals[section][type][key]
		except Exception:
			return

	def lastVal(self, section, type, key, val=None):
		Log.logMsg(5,"Entering lastVal %s %s %s %s" % (section, type, key, val))
		if not section in self.lastVals:
			self.lastVals[section] = {}
		if not type in self.lastVals[section]:
			self.lastVals[section][type] = {}
		if val is not None:
			self.lastVals[section][type][key] = val
		if key in self.lastVals[section][type]:
			Log.logMsg(5,"returning lastVal %s %s %s %s" % (section, type, key, self.lastVals[section][type][key]))
			return self.lastVals[section][type][key]
		return None

	def compareKeys(self):
		stoppedservices = 0
		totalservices = 0
		for service in self.curServices():
			totalservices += 1
			if not self.lookUpConfig("SERVICE", service):
				stoppedservices += 1

  		if (stoppedservices == totalservices) and (totalservices > 1):
			raise Exception, "All services detected disabled."

		for sn in self.mtaconfig.getSections():
			section = self.mtaconfig.getSection(sn)
			Log.logMsg(5, "Checking keys for %s" % (section.name,))
			if len(self.forcedconfig):
				Log.logMsg(4, "Checking for forced keys %s" % (section.name,))
				if not section.name in self.forcedconfig:
					continue
			
			section.changed = False
			self.resetChangedKeys(section.name)

			for key in section.requiredvars():
				type = section.requiredvars(key)
				if re.match(r'!', key):
					key = re.sub(r'^!','',key)
				prev = self.lastVal(section.name,type,key)
				val = self.lookUpConfig(type, key)
				Log.logMsg(5, "Checking %s=%s" % (key,val))
				if val is not None:
					if prev != val:
						if not self.firstRun:
							Log.logMsg(3, "Var %s changed from \'%s\' -> \'%s\'" % (key, prev, val))
						self.lastVal(section.name, type, key, val)
						self.changedKeys(section.name,key)
						section.changed = True
				else:
					Log.logMsg(5, "Required key is not defined %s=\'%s\'" % (key, val))
					if prev is not None:
						if not self.firstRun:
							Log.logMsg(3, "Var %s changed from \'%s\' to no longer defined." % (key, prev))
						self.delVal(section.name, type, key)
						section.changed = True
		
		for service in self.curServices():
			if not self.lookUpConfig("SERVICE", service):
				Log.logMsg(2, "service %s was disabled need to stop" % (service,))
				self.curRestarts(service,0)

  		for service in self.serverconfig.getServices():
			if self.curServices(service) is None:
				if self.firstRun:
					if (self.serverconfig.getServices(service)):
						self.curServices(service,"running")
					else:
						self.curServices(service,"stopped")
				else:
					Log.logMsg(2, "service %s was enabled need to start" % (service,))
					self.curRestarts(service, 1)


	def compileActions(self):
		for sn in self.mtaconfig.getSections():
			section = self.mtaconfig.getSection(sn)
			Log.logMsg(5, "compiling actions for %s" % (section.name,))
			if len(self.forcedconfig) or len(self.requestedconfig):
				Log.logMsg(4, "Checking for forced keys %s" % (section.name,))
				if not section.name in self.forcedconfig and not section.name in self.requestedconfig:
					continue
			
			if self.firstRun or section.changed or section.name in self.forcedconfig or section.name in self.requestedconfig:
				Log.logMsg(5, "Section %s changed compiling rewrites" % (section.name,))
				for rewrite in section.rewrites():
					self.curRewrites(rewrite, section.rewrites(rewrite))

				Log.logMsg(5, "Section %s changed compiling postconf" % (section.name,))
				for postconf in section.postconf():
					self.curPostconf(postconf, section.postconf(postconf))

				Log.logMsg(5, "Section %s changed compiling postconfd" % (section.name,))
				for postconfd in section.postconfd():
					self.curPostconfd(postconfd, section.postconfd(postconfd))

				if section.name == "proxy":
					Log.logMsg(5, "Section %s changed compiling proxygen" % (section.name,))
					self.proxygen(True)

				Log.logMsg(5, "Section %s changed compiling ldap" % (section.name,))
				for ldap in section.ldap():
					self.curLdap(ldap, section.ldap(ldap))

				if not self.forced and not self.firstRun and not len(self.requestedconfig): # no restarts on forced rewrites
					Log.logMsg(5, "Section %s changed compiling restarts" % (section.name,))
					for restart in section.restarts():
						if self.lookUpConfig("SERVICE", restart):
							Log.logMsg(5, "Adding restart %s" % (restart,))
							self.curRestarts(restart, -1)
						else:
							if restart == "archiving" and not self.serverconfig.getServices(restart):
								Log.logMsg(5, "%s not enabled, skipping stop" % (restart,))
							elif restart == "opendkim" and self.serverconfig.getServices("mta"):
								Log.logMsg(5, "Adding restart opendkim")
								self.curRestarts(restart, -1)
							else:
								Log.logMsg(5, "Adding stop %s" % (restart,))
								self.curRestarts(restart, 0)
			else:
				Log.logMsg(4, "Section %s did not change skipping" % (section.name,));


	def doProxygen(self):
		c = commands.commands["proxygen"]
		try:
			rc = c.execute((self.hostname,))
		except Exception, e:
			rc = 1
			[Log.logMsg(1,t) for t in traceback.format_tb(sys.exc_info()[2])]
			Log.logMsg(1, "Proxy configuration failed (%s)" % (e,))
		return rc

	def doRewrites(self):
		for (rewrite,(val,mode)) in self.curRewrites().items():
			# (val, mode) = self.curRewrites(rewrite)
			if not self.rewriteConfig(rewrite, val, mode):
				self.delRewrite(rewrite)

	def doPostconf(self):
		if self.curPostconf():
			c = commands.commands["postconf"]
			for (postconf, val) in self.curPostconf().items():
				try:
					rc = c.execute("%s='%s'" % (postconf, val))
				except Exception, e:
					return rc
			self.clearPostconf()
			return rc
		return 0

	def doPostconfd(self):
		if self.curPostconfd():
			c = commands.commands["postconfd"]
			for (postconfd, val) in self.curPostconfd().items():
				try:
					rc = c.execute("%s" % postconfd)
				except Exception, e:
					return rc
			self.clearPostconfd()
			return rc
		return 0

	def runProxygen(self):
		if self.proxygen():
			if not self.doProxygen():
				self.proxygen(False)

	def runLdap(self):
		for (ldap,val) in self.curLdap().items():
			if not self.doLdap(ldap, val):
				self.delLdap(ldap)

	def doConfigRewrites(self):
		t1 = time.clock()
		th = []
		# Proxygen takes longest, do it first
		th.append(threading.Thread(target=State.runProxygen,args=(self,),name="proxygen"))
		th.append(threading.Thread(target=State.doRewrites,args=(self,),name="rewrites"))
		th.append(threading.Thread(target=State.doPostconf,args=(self,),name="postconf"))
		th.append(threading.Thread(target=State.doPostconfd,args=(self,),name="postconfd"))
		th.append(threading.Thread(target=State.runLdap,args=(self,),name="ldap"))

		[t.start() for t in th]
		[t.join() for t in th]
		dt = time.clock()-t1
		Log.logMsg(3, "All rewrite threads completed in %.2f sec" % dt)

	def doRestarts(self):
		t1 = time.clock()
		# key=service, val=num-failed-restarts
		failedRestarts = {};
		# Don't thread these
		while self.curRestarts().items():	# Loop to pick up any dependencies.
			Log.logMsg(5, "doRestarts: curRestarts().items()=%s" % self.curRestarts().items())
			for (restart, val) in sorted (self.curRestarts().items(), key=lambda x: State.startorder[x[0]]):
				Log.logMsg(5, "doRestarts: restart=%s, val=%s" % (restart, val))
				if not self.controlProcess(restart, val):
					Log.logMsg(5, "doRestarts: deleting successful restart=%s, val=%s" % (restart, val))
					self.delRestart(restart)
				else:
					Log.logMsg(5, "doRestarts: restart failed, restart=%s, val=%s" % (restart, val))
					if failedRestarts.setdefault(restart, 0) >= self.maxFailedRestarts:
						Log.logMsg(2, "doRestarts: deleting excessive restart=%s, val=%s" % (restart, val))
						self.delRestart(restart)
					else:
						failedRestarts[restart] += 1
						# prevent thrashing
						time.sleep(1)

		dt = time.clock()-t1
		Log.logMsg(3, "All restarts completed in %.2f sec" % dt)

	def doLdap(self, key, val):
		Log.logMsg(4, "Setting ldap %s=%s" % (key, val))
		rc = 0
		try:
			rc = self.ldap.modify_attribute(key, val)
		except Exception, e:
				Log.logMsg(1, "LDAP FAILURE (%s)" % e)
		return rc

	def processIsRunning(self,process):
		return (not self.controlProcess(process, 2))

	def processIsNotRunning(self,process):
		return (not self.processIsRunning(process))

	def controlProcess(self, process, action_value):
		process = process.lower()
		rc = 0
		if not action_value in (-1, 0, 1, 2):
			Log.logMsg(1,"controlProcess %s (%s)" % (process, str(action_value)))
			Log.logMsg(1, "State must be in -1,0,1,2")
			return
		if not process in commands.commands:
			Log.logMsg(1, "Command not defined for %s" % (process,));
			return
		action = ["restart","stop","start","status"][action_value+1]
		lvl = 3
		if action == "status":
			lvl = 4

		Log.logMsg(lvl,"controlProcess %s %s (%d)" % (process, action, action_value))

		# return if it's already running and we are trying to start it.

		if action == "start" and self.processIsRunning(process):
			self.curServices(process,"running")
			Log.logMsg(lvl,"%s was already running adding to current state." % (process,))
			self.compileDependencyRestarts(process);
			return 0

		# if we initiate a stop/restart remove the service from watchdog
		# list of services available for restarts.  This avoids a restart
		# loop if they are slow to startup.
		if action in ["stop", "restart"] and self.getWatchdog(process):
			self.delWatchdog(process)

		rewrite = "norewrite"

		# Postfix, unique to the end.
		if process == "mta" and action == "restart":
			action = "reload"

		Log.logMsg(lvl,"CONTROL %s: %s %s %s" % (process, commands.exe[process.upper()], action, rewrite) )
		if action != "status":
			# log at lvl 1 to make sure it gets into syslog
			Log.logMsg(1, "%s %s initiated from zmconfigd" % (process, action))

		pargs = ' '.join([action,rewrite])
		try:
			rc = commands.commands[process].execute((pargs,))
		except Exception, e:
			Log.logMsg(1,"Exception in %s: (%s)" % (commands.exe[process.upper()], e) )
			rc = 1

		if rc == 0:
			if action == "stop":
				Log.logMsg(2, "%s was stopped removing from current state" % (process,));
				self.compileDependencyRestarts(process)
				if self.curServices(process):
					del(self.current["services"][process])
			elif action == "start":
				Log.logMsg(2, "%s was started adding to current state" % (process,));
				self.compileDependencyRestarts(process)
				self.curServices(process, "started")
		elif action != "status":
			Log.logMsg(2, "Failed to %s %s rc=%d" % (action, process, rc))

		return rc

	def rewriteConfig(self, fr, to, mode=None):
		t1 = time.clock()
		mode = mode or '0440'
		# This converts what may or may not be a string to an int
		mode = eval(str(mode))

		# Automatically handles absolute paths
		fr = os.path.join(self.baseDir,fr)
		to = os.path.join(self.baseDir,to)
		Log.logMsg(5, "Rewriting %s -> %s (%o)" % (fr, to, mode));

		try:
			(fh, tmpfile) = tempfile.mkstemp(dir="/tmp")

			f = fh.asOutputStream()
			for line in open(fr).readlines():
				f.write(self.transform(line))

			f.close()
			os.chmod(tmpfile, mode)
			# Can't find an atomic clobber move in jython
			if os.path.exists(to):
				os.unlink(to)
			shutil.move(tmpfile, to)
			dt = time.clock() - t1
			Log.logMsg(3, "Rewrote: %s with mode %o (%.2f sec)" % (to, mode, dt))
		except Exception, e:
			[Log.logMsg(1,t) for t in traceback.format_tb(sys.exc_info()[2])]
			Log.logMsg(1, "Rewrite failed: %s (%s)" % (e))
			return True

		return False

	def xformLocalConfig(self, match):
		sr = match.group(1)
		func = None
		key = sr
		if re.search(r' ',sr):
			(func, key) = sr.split(' ',1)
		Log.logMsg(5, "xformLocalConfig %s" % (sr,))
		val = self.localconfig[key]
		if func:
			if func == "SPLIT":
				val = val.split(' ',1)[0]
			elif func == "PERDITION_LDAP_SPLIT":
				# This appears to be legacy
				# we need first arg plus just host names from remaining args
				val = ' '.join((val.split()[0],' '.join(re.findall(r"ldap.?://(\S*):\d+",val))))
				Log.logMsg(5, "PERDITION_LDAP_SPLIT: %s" % (val,))

		if val is None:
			val = ""
		return val

	# We support parsing for the zmprov -l functions.
	# Normal parsing uses gcf
	# Functions supported:
	#  (un)comment(args) - replace with comment char "#" if true (or value exists)
	#   binary(args) - 0 for false, 1 for true
	#   range (var low high) - replace with percent of range
	#   freq (var total) - replace with total / var  (var is period in total)
	#   contains (var string) - 
	#    for MV attribs, set to string if string is in the attrib
	#   contains (var string, replacement) - 
	#    for MV attribs, set to replacement if string is in the attrib
	#   exact (var string) - 
	#    for MV attribs, set to string if a value of attrib exactly matches string
	#   contains (var string, replacement) - 
	#    for MV attribs, set to replacement if a value of attrib exactly matches string
	#   list (var separator)
	#    Works like perl join, for multivalued attrib, joins with join value
	#    used to create csv or regexes
	#   truefalse
	#   explode
	#
	# args supported:
	#  SERVER:key - use command gs with zimbra_server_hostname, get value of key
	#

	def xformConfigVariable(self, match):
		sr = match.group(1)
		val = None
		val = self.lookUpConfig("VAR", sr)
		if val is None:
			val = self.lookUpConfig("LOCAL", sr)

		# Requires a string return for re.sub()
		if val is None:
			val = ""
		return str(val)

	def xformConfig(self, match):
		sr = match.group(1)
		val = None

		if re.match(r"comment", sr):
			[(cmd,key)] = re.findall(r"comment ([^:]+):(\S+)",sr)
			Log.logMsg(5, "comment before lookup key=%s cmd=%s sr=%s" % (key, cmd, sr))

			parts = key.split(',',2)
			commentstr = '#'
			if (len(parts) > 1):
				key = parts[0]
				commentstr = parts[1]
			val = self.lookUpConfig(cmd, key)
			Log.logMsg(5, "comment after lookup key=%s val=%s cmd=%s sr=%s" % (key, val, cmd, sr))
			if (len(parts) > 2):
				valset = parts[2].split(',')
				if val is None:
					val = commentstr
				elif val in valset:
					val = commentstr
				else:
					val = ""
			else:
				if self.isTrueValue(val):
					val = commentstr
				else:
					val = ""
			Log.logMsg(5, "comment after rep key=%s val=%s cmd=%s" % (key, val, cmd))

		elif re.match(r"uncomment", sr):
			[(cmd,key)] = re.findall(r"uncomment ([^:]+):(\S+)",sr)
			Log.logMsg(5, "uncomment before lookup key=%s cmd=%s sr=%s" % (key, cmd, sr))

			parts = key.split(',',2)
			commentstr = '#'
			if (len(parts) > 1):
				key = parts[0]
				commentstr = parts[1]
			val = self.lookUpConfig(cmd, key)
			Log.logMsg(5, "uncomment after lookup key=%s val=%s cmd=%s sr=%s" % (key, val, cmd, sr))
			if (len(parts) > 2):
				valset = parts[2].split(',')
				if val is None:
					val = ""
				elif val in valset:
					val = ""
				else:
					val = commentstr
			else:
				if self.isTrueValue(val):
					val = ""
				else:
					val = commentstr
			Log.logMsg(5, "uncomment after rep key=%s val=%s cmd=%s" % (key, val, cmd))

		elif re.match(r"binary", sr):
			[(cmd,key)] = re.findall(r"binary ([^:]+):(\S+)",sr)
			val = 0
			if self.isTrueValue(self.lookUpConfig(cmd, key)):
				val = 1

		elif re.match(r"truefalse", sr):
			[(cmd,key)] = re.findall(r"truefalse ([^:]+):(\S+)",sr)
			Log.logMsg(5, "%s %s %s" % (cmd, key, val));
			val = "false"
			if self.isTrueValue(self.lookUpConfig(cmd, key)):
				val = "true"
			Log.logMsg(5, "%s %s %s" % (cmd, key, val));

		elif re.match(r"range", sr):
			[(cmd,key,lo,hi)] = re.findall(r"range ([^:]+):(\S+)\s+(\S+)\s+(\S+)",sr)
			val = int(self.lookUpConfig(cmd,key))
			val = ((val/100.00) * (int(hi) - int(lo))) + int(lo)

		elif re.match(r"list", sr):
			fields = sr.split(' ',2)
			(type,key) = fields[1].split(':')
			val = self.lookUpConfig(type, key)
			if val:
				val = fields[2].join(val.split())
			else:
				val = ""

		elif re.match(r"contains", sr):
			f = sr.split('^',2)
			st = f[0]
			if len(f) > 2:
				replace = f[1].strip()
				altreplace = f[2].strip()
			elif len(f) > 1:
				replace = f[1].strip()
				altreplace = ""
			else:
				replace = ""
				altreplace = ""
			fields = st.split(' ',2)
			(type,key) = fields[1].split(':')
			val = str(self.lookUpConfig(type, key))
			replace = replace or fields[2]
			Log.logMsg(5, "debug contains: if type %s for key=%s contains %s replace=%s or altreplace=%s" % (type, key, fields[2], replace, altreplace))
			if fields[2] in val:
				val = replace
			else: 
				val = altreplace
			Log.logMsg(5, "contains: type=%s key=%s val=%s" % (type, key, val))

		elif re.match(r"exact", sr):
			f = sr.split('^',2)
			st = f[0]
			if len(f) > 2:
				replace = f[1].strip()
				altreplace = f[2].strip()
			elif len(f) > 1:
				replace = f[1].strip()
				altreplace = ""
			else:
				replace = ""
				altreplace = ""
			fields = st.split(' ',2)
			(type,key) = fields[1].split(':')
			val = self.lookUpConfig(type, key)
			if val is None:
				val = ""
				return str(val)
			else:
				val = val.split()
			replace = replace or fields[2]
			Log.logMsg(5, "debug exact: type %s for key=%s exact matches %s replace=%s or altreplace=%s" % (type, key, fields[2], replace, altreplace))
			if fields[2] in val:
				val = replace
			else: 
				val = altreplace
			Log.logMsg(5, "exact: type=%s key=%s val=%s" % (type, key, val))

		elif re.match(r"freq", sr):
			[(cmd,key,total)] = re.findall(r"freq ([^:]+):(\S+)\s+(\S+)",sr)
			val = self.lookUpConfig(cmd,key)
			per = re.sub(r"\d+","", val)
			val = re.sub(r"\D","", val)
			val = int(val)
			total = int(total)

			if per == "m":
				val = val / 60
			elif per == "s":
				val = val / 3600
			elif per == "d":
				val = val * 24
			
			if val:
				val = int(total/val)
			else:
				val = total

			if val < 1 and total > 1:
				val = 1

		elif re.match(r"explode", sr):
			[(base,cmd,key)] = re.findall(r"explode (.*) ([^:]+):(\w+)",sr)
			Log.logMsg (5, "Explode %s" % (sr,))
			vals = self.lookUpConfig(cmd, key)
			val = []
			if vals:
				for v in vals.split():
					qr = "%s:%s" % (cmd,key)
					Log.logMsg(5, "Substituting %s in %s with %s" % (qr,sr,v));
					Log.logMsg(5, "Quoted string=%s" % (qr,));
					Log.logMsg(5, "sline=%s" % (match.group(0)));
					val.append("%s %s" % (base, v))
					Log.logMsg(5, "Final string=%s" % (val[-1],))
				val = '\n'.join(val)
			else:
				val = ''
		else:
			val = self.lookUpConfig("VAR", sr)
			if val is None:
				val = self.lookUpConfig("LOCAL", sr)

		# Requires a string return for re.sub()
		if val is None:
			val = ""
		return str(val)

	def transform(self, line):
		if(line.count('@') < 2 and line.count('%') < 2):
			return line

		line = re.sub(r"@@([^@]+)@@", self.xformLocalConfig, line)

		# If the line begins and ends with %%, then we are asking a special action be done
		# However, before we do that action, we need to do variable substitution in the line
		# and then evaluate the action
		line = line.rstrip()

		if(line.startswith("%%contains") and line.endswith("%%") and line.count('%') > 4):
			line = re.sub('^%%','',line)
			line = re.sub('%%$','',line)
			line = re.sub(r"%%([^%]+)%%", self.xformConfigVariable, line)
			line = line + "%%"
			line = "%%" + line
		line = line + '\n'
		line = re.sub(r"%%([^%]+)%%", self.xformConfig, line)
		return line

	def compileDependencyRestarts(self, name):
		section = self.mtaconfig.getSection(name)
		if (section is not None):
			for depend in section.depends():
				if self.lookUpConfig("SERVICE", depend) or depend == "amavis":
					Log.logMsg(3, "Adding restart for dependency %s" % (depend,));
					self.curRestarts(depend, -1)
